/*
 * encrypt ~ a simple, modular, (multi-OS) encryption utility
 * Copyright © 2005-2025, albinoloverats ~ Software Development
 * email: encrypt@albinoloverats.net
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

package net.albinoloverats.android.encrypt.lib.crypt;

import android.app.Service;
import android.content.Intent;
import android.net.Uri;
import androidx.documentfile.provider.DocumentFile;
import gnu.crypto.mode.ModeFactory;
import gnu.crypto.prng.LimitReachedException;
import lombok.val;
import net.albinoloverats.android.encrypt.lib.io.EncryptedFileInputStream;
import net.albinoloverats.android.encrypt.lib.misc.Convert;
import org.tukaani.xz.XZFormatException;
import org.tukaani.xz.XZInputStream;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.util.Arrays;
import java.util.HashMap;

import static net.albinoloverats.android.encrypt.lib.IntentKey.CIPHER;
import static net.albinoloverats.android.encrypt.lib.IntentKey.ENCRYPTING;
import static net.albinoloverats.android.encrypt.lib.IntentKey.HASH;
import static net.albinoloverats.android.encrypt.lib.IntentKey.KDF_ITERATIONS;
import static net.albinoloverats.android.encrypt.lib.IntentKey.MAC;
import static net.albinoloverats.android.encrypt.lib.IntentKey.MODE;

public class Decrypt extends Crypto
{
	private static final String SELF = ".";
	private static final String DEFAULT_OUTPUT_FILENAME = "decrypted";
	public static final String MIME_TYPE = "application/octet-stream";

	private boolean symlinkWarning = false;

	@Override
	public int onStartCommand(final Intent intent, final int flags, final int startId)
	{
		if (intent == null)
			return Service.START_REDELIVER_INTENT;

		preInit();

		val source = getSource(intent).get(0);
		val output = getOutput(intent);

		try
		{
			contentResolver = getContentResolver();
			this.source = new EncryptedFileInputStream(contentResolver.openInputStream(source));

			val documentFile = DocumentFile.fromSingleUri(this, output);
			if (documentFile == null)
				throw new IOException("");
			name = documentFile.getName();

			if (documentFile.exists() && !documentFile.isDirectory())
				this.output = contentResolver.openOutputStream(output);
			path = documentFile.getUri();
		}
		catch (final IOException e)
		{
			status = Status.FAILED_IO;
		}
		if (raw)
		{
			cipher = intent.getStringExtra(CIPHER.name());
			hash = intent.getStringExtra(HASH.name());
			mode = intent.getStringExtra(MODE.name());
			mac = intent.getStringExtra(MAC.name());
			kdfIterations = intent.getIntExtra(KDF_ITERATIONS.name(), KDF_ITERATIONS_DEFAULT);
		}

		intent.putExtra(ENCRYPTING.name(), false);
		return super.onStartCommand(intent, flags, startId);
	}

	@Override
	protected void process() throws CryptoProcessException
	{
		try
		{
			status = Status.RUNNING;

			version = raw ? Version.CURRENT : readVersion();
			if (status != Status.RUNNING)
				throw new Exception("Could not parse header!");

			var extraRandom = true;
			var ivType = XIV.RANDOM;
			var useMAC = true;
			switch (version)
			{
				case _201108:
				case _201110:
					ivType = XIV.BROKEN;
				case _201211:
					extraRandom = false;
					useMAC = false;
					break;
				case _201302:
				case _201311:
				case _201406:
					ivType = XIV.SIMPLE;
					useMAC = false;
					break;
				case _201501:
				case _201510:
					useMAC = false;
					break;
				case _201709:
					kdfIterations = KDF_ITERATIONS_201709;
					break;
				case _202001:
				case _202201:
				case _202401:
				case CURRENT:
					//kdfIterations = KDF_ITERATIONS_DEFAULT;
					break;
			}

			verification = ((EncryptedFileInputStream)source).initialiseDecryption(cipher, hash, mode, mac, kdfIterations, key, ivType, useMAC);

			if (!raw)
			{
				if (extraRandom)
					skipRandomData();
				readVerificationSum();
				skipRandomData();
			}
			readMetadata();
			if (extraRandom && !raw)
				skipRandomData();

			source = compressed ? new XZInputStream(source) : source;

			verification.hash().reset();

			if (directory)
				decryptDirectory(path);
			else
			{
				current.size = total.size;
				total.size = 1;
				if (blockSize > 0)
					decryptStream();
				else
					decryptFile();
			}

			if (version != Version._201108 && !raw)
			{
				val digest = verification.hash().digest();
				val check = new byte[verification.hash().hashSize()];
				val err = readAndHash(check);
				if (err < 0 || !Arrays.equals(check, digest))
					status = Status.WARNING_CHECKSUM;
			}
			if (!raw)
				skipRandomData();
			if (useMAC && version.compareTo(Version._202001) >= 0)
			{
				val digest = verification.mac().digest();
				val check = new byte[verification.mac().macSize()];
				val err = readAndHash(check);
				if (err < 0 || !Arrays.equals(check, digest))
					status = Status.WARNING_CHECKSUM;
			}

			if (status == Status.RUNNING)
				status = Status.SUCCESS;
			if (symlinkWarning)
				status = Status.WARNING_LINK;
		}
		catch (final CryptoProcessException e)
		{
			status = e.code;
			throw e;
		}
		catch (final NoSuchAlgorithmException e)
		{
			status = Status.FAILED_UNKNOWN_ALGORITHM;
			throw new CryptoProcessException(Status.FAILED_UNKNOWN_ALGORITHM, e);
		}
		catch (final InvalidKeyException | LimitReachedException e)
		{
			status = Status.FAILED_OTHER;
			throw new CryptoProcessException(Status.FAILED_OTHER, e);
		}
		catch (final XZFormatException e)
		{
			status = Status.FAILED_COMPRESSION_ERROR;
			throw new CryptoProcessException(Status.FAILED_COMPRESSION_ERROR, e);
		}
		catch (final IOException e)
		{
			status = Status.FAILED_IO;
			throw new CryptoProcessException(Status.FAILED_IO, e);
		}
		catch (final Throwable t)
		{
			if (status == Status.RUNNING)
				status = Status.FAILED_OTHER;
			throw new CryptoProcessException(status, t);
		}
		finally
		{
			closeIgnoreEverything(source);
			closeIgnoreEverything(output);
		}
	}

	private Version readVersion() throws CryptoProcessException, IOException
	{
		val header = new byte[Long.SIZE / Byte.SIZE];
		for (int i = 0; i < HEADER.length; i++)
			if (source.read(header, 0, header.length) < 0)
				throw new IOException("Could not read data");

		val v = Version.parseMagicNumber(Convert.longFromBytes(header), null);
		if (v == null)
			throw new CryptoProcessException(Status.FAILED_UNKNOWN_VERSION);

		if (v.compareTo(Version._201510) >= 0 && !raw)
			((EncryptedFileInputStream)source).initialiseECC();

		val b = new byte[source.read()];
		read(source, b);

		val a = new String(b);
		cipher = a.substring(0, a.indexOf('/'));
		hash = a.substring(a.indexOf('/') + 1);
		if (hash.contains("/"))
		{
			mode = hash.substring(hash.indexOf('/') + 1);
			hash = hash.substring(0, hash.indexOf('/'));
			if (mode.contains("/"))
			{
				mac = mode.substring(mode.indexOf('/') + 1);
				mode = mode.substring(0, mode.indexOf('/'));
				if (mac.contains("/"))
				{
					val kdf = mac.substring(mac.indexOf('/') + 1);
					mac = mac.substring(0, mac.indexOf('/'));
					/*
					 * tough tits if you used more than 2,147,483,647 iterations
					 * on your desktop (where libgcrypt uses an unsigned long)
					 */
					kdfIterations = (int)Long.parseLong(kdf, 0x10);
				}
			}
		}
		else
			mode = ModeFactory.CBC_MODE;

		return v;
	}

	private void readVerificationSum() throws CryptoProcessException, IOException
	{
		val buffer = new byte[Long.SIZE / Byte.SIZE];
		readAndHash(buffer);
		val x = Convert.longFromBytes(buffer);
		readAndHash(buffer);
		val y = Convert.longFromBytes(buffer);
		readAndHash(buffer);
		val z = Convert.longFromBytes(buffer);
		if ((x ^ y) != z)
			throw new CryptoProcessException(Status.FAILED_DECRYPTION);
	}

	private void readMetadata() throws CryptoProcessException, IOException
	{
		val c = new byte[1];
		readAndHash(c);
		for (int i = 0; i < (short)(Convert.byteFromBytes(c) & 0x00FF); i++)
		{
			val tv = new byte[1];
			readAndHash(tv);
			val tag = Tag.fromValue((short)(Convert.byteFromBytes(tv) & 0x00FF));
			val l = new byte[Short.SIZE / Byte.SIZE];
			readAndHash(l);
			val length = Convert.shortFromBytes(l);
			val v = new byte[length];
			readAndHash(v);
			switch (tag)
			{
				case SIZE:
					total.size = Convert.longFromBytes(v);
					break;
				case BLOCKED:
					blockSize = (int)Convert.longFromBytes(v);
					break;
				case COMPRESSED:
					compressed = Convert.booleanFromBytes(v);
					break;
				case DIRECTORY:
					directory = Convert.booleanFromBytes(v);
					break;
				case FILENAME:
					name = new String(v);
					break;
			}
		}
		if (!directory && name == null)
			name = DEFAULT_OUTPUT_FILENAME;
		val documentFile = DocumentFile.fromTreeUri(this, path);
		if (documentFile == null)
		{
			status = Status.FAILED_IO;
			return;
		}
		contentResolver.takePersistableUriPermission(path, Intent.FLAG_GRANT_READ_URI_PERMISSION | Intent.FLAG_GRANT_WRITE_URI_PERMISSION);
		if (!directory)
		{
			val file = documentFile.createFile(MIME_TYPE, name);
			if (file == null)
				status = Status.FAILED_IO;
			else
				output = contentResolver.openOutputStream(file.getUri());
		}
	}

	private void skipRandomData() throws IOException
	{
		val b = new byte[1];
		readAndHash(b);
		readAndHash(new byte[(short)(Convert.byteFromBytes(b) & 0x00FF)]);
	}

	private void decryptDirectory(final Uri uri) throws CryptoProcessException, IOException
	{
		val directories = new HashMap<String, DocumentFile>();

		contentResolver.takePersistableUriPermission(uri, Intent.FLAG_GRANT_READ_URI_PERMISSION | Intent.FLAG_GRANT_WRITE_URI_PERMISSION);
		val root = DocumentFile.fromTreeUri(this, uri);

		directories.put(SELF, root);

		for (total.offset = 0; total.offset < total.size && status == Status.RUNNING; total.offset++)
		{
			var b = new byte[1];
			readAndHash(b);
			val t = FileType.fromID(b[0]);
			b = new byte[Long.SIZE / Byte.SIZE];
			readAndHash(b);
			var l = Convert.longFromBytes(b);
			b = new byte[(int)l];
			readAndHash(b);
			val fullPath = new String(b);
			val path = new File(fullPath).toPath();
			var parent = directories.get(path.getParent() != null ? path.getParent().toString() : SELF);
			if (parent == null)
				continue;

			switch (t)
			{
				case DIRECTORY:
					var p = "";
					for (val d : fullPath.split("/"))
					{
						p += d;
						if (!directories.containsKey(p))
						{
							parent = parent.createDirectory(d);
							if (parent == null)
								throw new IOException("Could not create directory: " + d);
							directories.put(p, parent);
						}
						p += '/';
					}
					break;
				case REGULAR:
					current.offset = 0;
					b = new byte[Long.SIZE / Byte.SIZE];
					readAndHash(b);
					val filename = path.getFileName().toString();
					current.file = filename;
					current.size = Convert.longFromBytes(b);
					val newFile = parent.createFile(MIME_TYPE, filename);
					if (newFile == null)
						throw new IOException("Could not create file: " + filename);
					output = contentResolver.openOutputStream(newFile.getUri());
					decryptFile();
					current.offset = current.size;
					output.close();
					output = null;
					break;
				case LINK:
				case SYMLINK:
					symlinkWarning = true;
					b = new byte[Long.SIZE / Byte.SIZE];
					readAndHash(b);
					l = Convert.longFromBytes(b);
					b = new byte[(int)l];
					readAndHash(b);
/*					val link = parent.getName() + File.separator + new String(b);
					val target = new File(link).toPath();
					if (t == FileType.LINK)
						Files.createLink(path, target);
					else
						Files.createSymbolicLink(path, target);*/
					break;
			}
		}
	}

	private void decryptStream() throws IOException
	{
		var b = true;
		var buffer = new byte[blockSize];
		while (b && status == Status.RUNNING)
		{
			b = source.read() == 1;
			read(source, buffer);
			int r = blockSize;
			if (!b)
			{
				val l = new byte[Long.SIZE / Byte.SIZE];
				read(source, l);
				r = (int)Convert.longFromBytes(l);
				val tmp = new byte[r];
				System.arraycopy(buffer, 0, tmp, 0, r);
				buffer = new byte[r];
				System.arraycopy(tmp, 0, buffer, 0, r);
			}
			verification.hash().update(buffer, 0, r);
			verification.mac().update(buffer, 0, r);
			output.write(buffer);
			current.offset += r;
		}
	}

	private void decryptFile() throws IOException
	{
		val buffer = new byte[BLOCK_SIZE];
		for (current.offset = 0; current.offset < current.size && status == Status.RUNNING; current.offset += BLOCK_SIZE)
		{
			int j = BLOCK_SIZE;
			if (current.offset + BLOCK_SIZE > current.size)
				j = (int)(BLOCK_SIZE - (current.offset + BLOCK_SIZE - current.size));
			val r = readAndHash(buffer, j);
			output.write(buffer, 0, r);
		}
	}

	private int readAndHash(final byte[] b) throws IOException
	{
		return readAndHash(b, b.length);
	}

	private int readAndHash(final byte[] b, final int l) throws IOException
	{
		val r = source.read(b, 0, l);
		verification.hash().update(b, 0, l);
		verification.mac().update(b, 0, l);
		return r;
	}

	private static void read(final InputStream in, final byte[] b) throws IOException
	{
		if (in.read(b) < 0)
			throw new IOException("Could not read data");
	}
}
