/*
 * encrypt ~ a simple, modular, (multi-OS) encryption utility
 * Copyright © 2005-2025, albinoloverats ~ Software Development
 * email: encrypt@albinoloverats.net
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

package net.albinoloverats.android.encrypt.lib.io;

import gnu.crypto.cipher.IBlockCipher;
import gnu.crypto.mac.IMac;
import gnu.crypto.mode.IMode;
import gnu.crypto.mode.ModeFactory;
import gnu.crypto.prng.IPBE;
import gnu.crypto.prng.LimitReachedException;
import gnu.crypto.prng.PBKDF2;
import gnu.crypto.util.PRNG;
import lombok.val;
import net.albinoloverats.android.encrypt.lib.crypt.CryptoUtils;
import net.albinoloverats.android.encrypt.lib.crypt.XIV;
import net.albinoloverats.android.encrypt.lib.misc.Convert;

import java.io.IOException;
import java.io.OutputStream;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.util.HashMap;
import java.util.Map;

public class EncryptedFileOutputStream extends OutputStream
{
	private final ECCFileOutputStream eccFileOutputStream;

	private IMode cipher;

	private byte[] buffer = null;
	private int blockSize = 0;
	private final int[] offset = { 0, 0 };

	private boolean open = true;

	public EncryptedFileOutputStream(final OutputStream stream)
	{
		eccFileOutputStream = new ECCFileOutputStream(stream);
	}

	public HashMAC initialiseEncryption(final String c, final String h, final String m, final String a, final int kdfIterations, final byte[] k, final XIV ivType, final boolean useKDF) throws NoSuchAlgorithmException, InvalidKeyException, LimitReachedException, IOException
	{
		val hash = CryptoUtils.getHashAlgorithm(h);
		val cipher = CryptoUtils.getCipherAlgorithm(c);
		val mac = CryptoUtils.getMacAlgorithm(a);

		blockSize = cipher.defaultBlockSize();
		buffer = new byte[blockSize];

		this.cipher = ModeFactory.getInstance(m, cipher, blockSize);
		hash.update(k, 0, k.length);
		val keySource = hash.digest();
		Map<String, Object> attributes;
		val keyLength = CryptoUtils.getCipherAlgorithmKeySize(c) / Byte.SIZE;
		var key = new byte[keyLength];

		val salt = new byte[keyLength];

		val keyMac = CryptoUtils.getMacAlgorithm(CryptoUtils.hmacFromHash(h));

		if (useKDF)
		{
			PRNG.nextBytes(salt);
			eccFileOutputStream.write(salt);

			var keyGen = new PBKDF2(keyMac);
			attributes = new HashMap<>();
			attributes.put(IMac.MAC_KEY_MATERIAL, keySource);
			attributes.put(IPBE.SALT, salt);
			attributes.put(IPBE.ITERATION_COUNT, kdfIterations);
			keyGen.init(attributes);
			keyGen.nextBytes(key);
		}
		else
			System.arraycopy(keySource, 0, key, 0, Math.min(keyLength, keySource.length));

		attributes = new HashMap<>();
		attributes.put(IBlockCipher.KEY_MATERIAL, key);
		attributes.put(IBlockCipher.CIPHER_BLOCK_SIZE, blockSize);
		attributes.put(IMode.STATE, IMode.ENCRYPTION);
		hash.reset();
		hash.update(keySource);
		val iv = new byte[ivType != XIV.BROKEN ? blockSize : keyLength];
		switch (ivType)
		{
			case BROKEN, SIMPLE:
				System.arraycopy(hash.digest(), 0, iv, 0, iv.length);
				break;
			case RANDOM:
				PRNG.nextBytes(iv);
				eccFileOutputStream.write(iv);
				break;
		}
		attributes.put(IMode.IV, iv);
		this.cipher.init(attributes);


		val macLength = CryptoUtils.getHashAlgorithm(CryptoUtils.hashFromHmac(a)).blockSize();
		key = new byte[macLength];
		val keyGen = new PBKDF2(keyMac);
		attributes = new HashMap<>();
		attributes.put(IMac.MAC_KEY_MATERIAL, keySource);
		attributes.put(IPBE.SALT, salt);
		attributes.put(IPBE.ITERATION_COUNT, kdfIterations);
		keyGen.init(attributes);
		keyGen.nextBytes(key);
		attributes = new HashMap<>();
		attributes.put(IMac.MAC_KEY_MATERIAL, key);
		mac.init(attributes);

		return new HashMAC(hash, mac);
	}

	public void initialiseECC()
	{
		eccFileOutputStream.initialise();
	}

	@Override
	public void close() throws IOException
	{
		if (!open)
			return;
		if (cipher != null)
		{
			val remainder = new int[] { 0, blockSize - offset[0] };
			val x = new byte[remainder[1]];
			PRNG.nextBytes(x);
			System.arraycopy(x, 0, buffer, offset[0], remainder[1]);
			val eBytes = new byte[blockSize];
			cipher.update(buffer, 0, eBytes, 0);
			eccFileOutputStream.write(eBytes);
		}
		eccFileOutputStream.close();
		open = false;
	}

	@Override
	public void write(final byte[] bytes) throws IOException
	{
		if (cipher == null)
		{
			eccFileOutputStream.write(bytes);
			return;
		}
		val remainder = new int[] { bytes.length, blockSize - offset[0] };
		offset[1] = 0;
		while (remainder[0] > 0)
		{
			if (remainder[0] < remainder[1])
			{
				System.arraycopy(bytes, offset[1], buffer, offset[0], remainder[0]);
				offset[0] += remainder[0];
				return;
			}
			System.arraycopy(bytes, offset[1], buffer, offset[0], remainder[1]);
			val eBytes = new byte[blockSize];
			cipher.update(buffer, 0, eBytes, 0);
			eccFileOutputStream.write(eBytes);
			offset[0] = 0;
			buffer = new byte[blockSize];
			offset[1] += remainder[1];
			remainder[0] -= remainder[1];
			remainder[1] = blockSize - offset[0];
		}
	}

	@Override
	public void write(final byte[] b, final int off, final int len) throws IOException
	{
		val bytes = new byte[len];
		System.arraycopy(b, off, bytes, 0, len);
		write(bytes);
	}

	@Override
	public void write(final int b) throws IOException
	{
		write(Convert.toBytes((byte)(b & 0x000000FF)));
	}
}
